import XCTest
@testable import EngineModule

/// KIF→SFEN変換のテスト
final class KifConversionTests: XCTestCase {
    
    func testProblemMoves() async throws {
        // 問題が発生している手順をテスト
        let moves = [
            "7g7f",  // 1手目
            "3c3d",  // 2手目
            "6g6f",  // 3手目
            "8b5b",  // 4手目
            "2h6h",  // 5手目
            "5c5d",  // 6手目
            "6i5h",  // 7手目
            "9c9d",  // 8手目
            "5i4h",  // 9手目
            "1c1d",  // 10手目
            "1g1f",  // 11手目
            "6a7b",  // 12手目
            "4h3h"   // 13手目 - ここでエラーが出ていた
        ]
        
        print("🧪 [KifConversionTests] 問題の手順をテスト")
        print("   手順: \(moves.joined(separator: " "))")
        
        // エンジンを初期化
        let coordinator = EngineCoordinator()
        
        do {
            try await coordinator.initialize(evalDir: nil)
            print("✅ [KifConversionTests] エンジン初期化成功")
        } catch {
            XCTFail("エンジン初期化失敗: \(error)")
            return
        }
        
        // 各手を順次テスト
        for (index, _) in moves.enumerated() {
            let movesUpToNow = moves[0...index].joined(separator: " ")
            let positionCommand = "startpos moves \(movesUpToNow)"
            
            print("📍 [KifConversionTests] \(index + 1)手目までテスト")
            
            do {
                let evaluation = try await coordinator.analyzePosition(
                    sfen: positionCommand,
                    timeLimit: 2.0
                )
                
                print("   ✅ 評価値: \(evaluation.score), 深さ: \(evaluation.depth)")
                
                // 評価値が0でないことを確認（エラーでない）
                if index > 5 {
                    XCTAssertNotEqual(evaluation.score, 0, "\(index + 1)手目で評価値が0（エラーの可能性）")
                }
                
            } catch {
                XCTFail("\(index + 1)手目でエラー: \(error)")
                print("   ❌ エラー: \(error)")
            }
        }
        
        print("🎉 [KifConversionTests] すべての手のテスト完了")
    }
    
    func testIndividualProblematicMoves() async throws {
        // 特に問題が出ていた手を個別にテスト
        let testCases: [(String, String)] = [
            ("7手目", "startpos moves 7g7f 3c3d 6g6f 8b5b 2h6h 5c5d 6i5h"),
            ("13手目", "startpos moves 7g7f 3c3d 6g6f 8b5b 2h6h 5c5d 6i5h 9c9d 5i4h 1c1d 1g1f 6a7b 4h3h"),
            ("14手目", "startpos moves 7g7f 3c3d 6g6f 8b5b 2h6h 5c5d 6i5h 9c9d 5i4h 1c1d 1g1f 6a7b 4h3h 7a6b"),
        ]
        
        let coordinator = EngineCoordinator()
        try await coordinator.initialize(evalDir: nil)
        
        for (label, position) in testCases {
            print("🧪 [KifConversionTests] \(label)をテスト")
            
            let evaluation = try await coordinator.analyzePosition(
                sfen: position,
                timeLimit: 2.0
            )
            
            print("   ✅ 評価値: \(evaluation.score), 深さ: \(evaluation.depth)")
            XCTAssertNotEqual(evaluation.score, 0, "\(label)で評価値が0")
        }
    }
    
    func testTimingIssues() async throws {
        // タイミングの問題をテスト
        print("🧪 [KifConversionTests] タイミング問題のテスト")
        
        let coordinator = EngineCoordinator()
        try await coordinator.initialize(evalDir: nil)
        
        // 連続して複数の局面を評価
        let positions = [
            "startpos moves 7g7f",
            "startpos moves 7g7f 3c3d",
            "startpos moves 7g7f 3c3d 6g6f",
        ]
        
        for (index, position) in positions.enumerated() {
            print("📍 [KifConversionTests] 連続評価 \(index + 1)/\(positions.count)")
            
            let evaluation = try await coordinator.analyzePosition(
                sfen: position,
                timeLimit: 1.0
            )
            
            print("   ✅ 評価値: \(evaluation.score)")
            XCTAssertNotEqual(evaluation.score, 0, "連続評価\(index + 1)で評価値が0")
            
            // 待機時間なし（タイミング問題を再現）
        }
        
        print("🎉 [KifConversionTests] タイミングテスト完了")
    }
}
