import XCTest
@testable import EngineModule

final class EngineIntegrationTests: XCTestCase {
    
    func testEngineInitialization() async throws {
        print("🧪 エンジン初期化テスト開始")
        
        let coordinator = EngineCoordinator()
        
        do {
            try await coordinator.initialize()
            print("✅ エンジン初期化成功")
            
            // クリーンアップ
            coordinator.shutdown()
        } catch {
            XCTFail("エンジン初期化失敗: \(error)")
        }
    }
    
    func testBasicAnalysis() async throws {
        print("🧪 基本解析テスト開始")
        
        let coordinator = EngineCoordinator()
        
        do {
            // 初期化
            try await coordinator.initialize()
            print("✅ エンジン初期化成功")
            
            // 初期局面を解析
            let startpos = "lnsgkgsnl/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL b - 1"
            print("📊 局面解析開始: \(startpos)")
            
            let result = try await coordinator.analyzePosition(sfen: startpos, timeLimit: 2.0)
            
            print("✅ 解析完了")
            print("  最善手: \(result.bestMove)")
            print("  評価値: \(result.score)")
            print("  探索深さ: \(result.depth)")
            print("  読み筋: \(result.pv.prefix(5).joined(separator: " "))")
            
            XCTAssertFalse(result.bestMove.isEmpty, "最善手が空")
            XCTAssertGreaterThan(result.depth, 0, "探索深さが0")
            
            // クリーンアップ
            coordinator.shutdown()
        } catch {
            coordinator.shutdown()
            XCTFail("解析失敗: \(error)")
        }
    }
    
    func testMultiPVAnalysis() async throws {
        print("🧪 MultiPV解析テスト開始")
        
        let coordinator = EngineCoordinator()
        
        do {
            // 初期化
            try await coordinator.initialize()
            print("✅ エンジン初期化成功")
            
            // 初期局面を解析（3つの候補手）
            let startpos = "lnsgkgsnl/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL b - 1"
            print("📊 MultiPV解析開始: \(startpos)")
            
            let candidates = try await coordinator.analyzeWithMultiPV(
                sfen: startpos,
                pvCount: 3,
                timeLimit: 3.0
            )
            
            print("✅ MultiPV解析完了")
            for (index, candidate) in candidates.enumerated() {
                print("  候補\(index + 1): \(candidate.moveUSI) (評価値: \(candidate.score), 深さ: \(candidate.depth))")
                print("    読み筋: \(candidate.pv.prefix(5).joined(separator: " "))")
            }
            
            XCTAssertGreaterThanOrEqual(candidates.count, 1, "候補手が1つ以上")
            XCTAssertLessThanOrEqual(candidates.count, 3, "候補手が3つ以下")
            
            // クリーンアップ
            coordinator.shutdown()
        } catch {
            coordinator.shutdown()
            XCTFail("MultiPV解析失敗: \(error)")
        }
    }
}
